<?php
/**
 * Plugin Name: SEO ALT Audit (Missing ALT)
 * Description: Zeigt Beiträge/Seiten mit <img>-Tags ohne ALT oder mit leerem ALT im WP-Admin (Dashboard Widget + Liste).
 * Version: 0.1.0
 * Author: Stefan Draeger
 */

if (!defined('ABSPATH')) exit;

class SEO_Alt_Audit {
    const CAPABILITY = 'manage_options';
    const MENU_SLUG  = 'seo-alt-audit';
    const CACHE_KEY  = 'seo_alt_audit_cache_v1';
    const CACHE_TTL  = 10 * MINUTE_IN_SECONDS;

    public function __construct() {
        add_action('wp_dashboard_setup', [$this, 'register_dashboard_widget']);
        add_action('admin_menu',         [$this, 'register_admin_page']);
        add_action('admin_post_seo_alt_audit_rescan', [$this, 'handle_rescan']);
    }

    public function register_dashboard_widget(): void {
        wp_add_dashboard_widget(
            'seo_alt_audit_widget',
            'SEO: Fehlende ALT-Attribute',
            [$this, 'render_dashboard_widget']
        );
    }

    public function register_admin_page(): void {
        add_menu_page(
            'SEO ALT Audit',
            'SEO ALT Audit',
            self::CAPABILITY,
            self::MENU_SLUG,
            [$this, 'render_admin_page'],
            'dashicons-search',
            80
        );
    }

    public function render_dashboard_widget(): void {
        if (!current_user_can(self::CAPABILITY)) {
            echo '<p>Keine Berechtigung.</p>';
            return;
        }

        $data = $this->get_scan_data();

        $posts_count = intval($data['counts']['post'] ?? 0);
        $pages_count = intval($data['counts']['page'] ?? 0);

        $url_posts = admin_url('admin.php?page=' . self::MENU_SLUG . '&type=post');
        $url_pages = admin_url('admin.php?page=' . self::MENU_SLUG . '&type=page');

        echo '<div style="line-height:1.6">';
        echo '<a href="' . esc_url($url_posts) . '"><strong>' . esc_html($posts_count) . '</strong> Beiträge haben Bilder mit fehlenden ALT Attribut.</a><br>';
        echo '<a href="' . esc_url($url_pages) . '"><strong>' . esc_html($pages_count) . '</strong> Seiten haben Bilder mit fehlenden ALT Attribut.</a>';
        echo '</div>';

        // Rescan Button
        $rescan_url = wp_nonce_url(
            admin_url('admin-post.php?action=seo_alt_audit_rescan'),
            'seo_alt_audit_rescan'
        );
        echo '<p style="margin-top:10px">';
        echo '<a class="button button-secondary" href="' . esc_url($rescan_url) . '">Jetzt neu scannen</a>';
        echo '</p>';
    }

    public function render_admin_page(): void {
        if (!current_user_can(self::CAPABILITY)) {
            wp_die('Keine Berechtigung.');
        }

        $type = isset($_GET['type']) && in_array($_GET['type'], ['post','page'], true) ? $_GET['type'] : 'post';
        $paged = max(1, intval($_GET['paged'] ?? 1));
        $per_page = 30;

        $data = $this->get_scan_data();
        $items = $data['items'][$type] ?? [];

        // Neueste zuerst (IDs desc)
        usort($items, function($a, $b) {
            return ($b['ID'] ?? 0) <=> ($a['ID'] ?? 0);
        });

        $total = count($items);
        $offset = ($paged - 1) * $per_page;
        $page_items = array_slice($items, $offset, $per_page);

        $base_url = admin_url('admin.php?page=' . self::MENU_SLUG . '&type=' . $type);
        $total_pages = max(1, (int)ceil($total / $per_page));

        echo '<div class="wrap">';
        echo '<h1>SEO ALT Audit</h1>';

        echo '<p>Gefunden werden <code>&lt;img&gt;</code>-Tags ohne <code>alt</code> oder mit leerem <code>alt</code> (z.B. <code>alt=""</code>).</p>';

        // Tabs
        echo '<h2 class="nav-tab-wrapper">';
        echo '<a class="nav-tab ' . ($type === 'post' ? 'nav-tab-active' : '') . '" href="' . esc_url(admin_url('admin.php?page=' . self::MENU_SLUG . '&type=post')) . '">Beiträge</a>';
        echo '<a class="nav-tab ' . ($type === 'page' ? 'nav-tab-active' : '') . '" href="' . esc_url(admin_url('admin.php?page=' . self::MENU_SLUG . '&type=page')) . '">Seiten</a>';
        echo '</h2>';

        // Rescan
        $rescan_url = wp_nonce_url(
            admin_url('admin-post.php?action=seo_alt_audit_rescan&redirect=' . rawurlencode($base_url)),
            'seo_alt_audit_rescan'
        );
        echo '<p><a class="button button-primary" href="' . esc_url($rescan_url) . '">Neu scannen</a></p>';

        // Table
        echo '<table class="widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th style="width:90px">ID</th>';
        echo '<th>Titel</th>';
        echo '<th style="width:120px">Fehlende ALTs</th>';
        echo '<th style="width:180px">Datum</th>';
        echo '<th style="width:120px">Aktion</th>';
        echo '</tr></thead>';
        echo '<tbody>';

        if (empty($page_items)) {
            echo '<tr><td colspan="5">Keine Einträge gefunden 🎉</td></tr>';
        } else {
            foreach ($page_items as $row) {
                $id = intval($row['ID']);
                $title = $row['post_title'] ?: '(ohne Titel)';
                $missing = intval($row['missing_alt_count']);
                $date = esc_html(mysql2date('Y-m-d H:i', $row['post_date'] ?? ''));

                $edit_link = get_edit_post_link($id, '');
                $view_link = get_permalink($id);

                echo '<tr>';
                echo '<td>' . esc_html($id) . '</td>';
                echo '<td>';
                echo '<strong><a href="' . esc_url($view_link) . '" target="_blank" rel="noopener noreferrer">' . esc_html($title) . '</a></strong>';
                echo '<div style="margin-top:4px; opacity:.85;">' . esc_html($view_link) . '</div>';
                echo '</td>';
                echo '<td><strong>' . esc_html($missing) . '</strong></td>';
                echo '<td>' . $date . '</td>';
                echo '<td><a class="button button-secondary" href="' . esc_url($edit_link) . '">Bearbeiten</a></td>';
                echo '</tr>';
            }
        }

        echo '</tbody>';
        echo '</table>';

        // Pagination
        if ($total_pages > 1) {
            echo '<div class="tablenav"><div class="tablenav-pages" style="margin: 12px 0;">';
            $prev = max(1, $paged - 1);
            $next = min($total_pages, $paged + 1);

            echo '<span class="displaying-num">' . esc_html($total) . ' Einträge</span> ';
            echo $paged > 1
                ? '<a class="button" href="' . esc_url($base_url . '&paged=' . $prev) . '">&laquo;</a> '
                : '<span class="button disabled">&laquo;</span> ';
            echo '<span style="padding:0 8px;">Seite ' . esc_html($paged) . ' von ' . esc_html($total_pages) . '</span>';
            echo $paged < $total_pages
                ? ' <a class="button" href="' . esc_url($base_url . '&paged=' . $next) . '">&raquo;</a>'
                : ' <span class="button disabled">&raquo;</span>';

            echo '</div></div>';
        }

        echo '</div>';
    }

    public function handle_rescan(): void {
        if (!current_user_can(self::CAPABILITY)) {
            wp_die('Keine Berechtigung.');
        }
        check_admin_referer('seo_alt_audit_rescan');

        // Cache löschen und neu erzeugen
        delete_transient(self::CACHE_KEY);
        $this->get_scan_data(true);

        $redirect = isset($_GET['redirect']) ? esc_url_raw($_GET['redirect']) : admin_url('index.php');
        wp_safe_redirect($redirect);
        exit;
    }

    private function get_scan_data(bool $force = false): array {
        if (!$force) {
            $cached = get_transient(self::CACHE_KEY);
            if (is_array($cached)) return $cached;
        }

        $result = [
            'counts' => ['post' => 0, 'page' => 0],
            'items'  => ['post' => [], 'page' => []],
        ];

        foreach (['post', 'page'] as $type) {
            $items = $this->scan_post_type($type);
            $result['items'][$type] = $items;
            $result['counts'][$type] = count($items);
        }

        set_transient(self::CACHE_KEY, $result, self::CACHE_TTL);
        return $result;
    }

    private function scan_post_type(string $type): array {
        global $wpdb;

        // Grobe Vorauswahl: Nur Inhalte mit <img ...> (Performance)
        // Hinweis: Das filtert noch nicht nach ALT, das machen wir danach robust per Parser.
        $rows = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT ID, post_title, post_date, post_content
                 FROM {$wpdb->posts}
                 WHERE post_type = %s
                   AND post_status = 'publish'
                   AND post_content LIKE %s",
                $type,
                '%<img%'
            ),
            ARRAY_A
        );

        $hits = [];

        foreach ($rows as $row) {
            $missing = $this->count_images_missing_alt($row['post_content'] ?? '');
            if ($missing > 0) {
                $hits[] = [
                    'ID' => intval($row['ID']),
                    'post_title' => $row['post_title'] ?? '',
                    'post_date' => $row['post_date'] ?? '',
                    'missing_alt_count' => $missing,
                ];
            }
        }

        return $hits;
    }

    private function count_images_missing_alt(string $html): int {
        if (stripos($html, '<img') === false) return 0;

        // Alle IMG-Tags extrahieren
        if (!preg_match_all('/<img\b[^>]*>/i', $html, $matches)) {
            return 0;
        }

        $count = 0;

        foreach ($matches[0] as $imgTag) {
            // ALT fehlt komplett?
            if (!preg_match('/\balt\s*=\s*([\'"])(.*?)\1/i', $imgTag, $altMatch)) {
                $count++;
                continue;
            }

            // ALT vorhanden aber leer/whitespace?
            $altValue = $altMatch[2] ?? '';
            if (trim(html_entity_decode($altValue)) === '') {
                $count++;
                continue;
            }
        }

        return $count;
    }
}

new SEO_Alt_Audit();
