/*
  ------------------------------------------------------------
  Mehr als vier Taster am ESP32 / Cheap Yellow Display
  ------------------------------------------------------------
  
  Beschreibung:
  In diesem Beispiel werden fünf Taster über eine Widerstands-
  leiter an nur einem einzigen ADC-Pin (GPIO35) des ESP32
  betrieben. Je nach gedrücktem Taster entsteht eine
  unterschiedliche Spannung am Analog-Eingang, die im Code
  ausgewertet wird.

  Die Erkennung erfolgt über definierte ADC-Wertebereiche
  (Fenster). Zusätzlich wird ein Mittelwert aus mehreren
  Messungen gebildet, um ADC-Schwankungen zu reduzieren.

  Wichtig:
  - Externer 10k Pullup-Widerstand erforderlich
  - Nur 3,3V verwenden (keine 5V am ADC!)
  - GPIO35 ist ein reiner INPUT-Pin

  Autor:
  Stefan Draeger

  Ausführlicher Blogartikel:
  https://draeger-it.blog/mehr-als-vier-taster-am-cheap-yellow-display-so-nutzt-du-nur-einen-gpio/

  ------------------------------------------------------------
*/

#define pin 35  // ADC-Pin (GPIO35 ist input-only)

struct Taster {
  int min_analog;
  int max_analog;
  const char* bezeichnung;
};

// ADC-Fenster (aus realen Messwerten ermittelt)
Taster taster[] = {
  { 1220, 1240, "Taster 1" },
  { 1780, 1800, "Taster 2" },
  { 2490, 2600, "Taster 3" },
  { 3100, 3120, "Taster 4" },
  { 3660, 3675, "Taster 5" }
};

const int ANZAHL_TASTER = sizeof(taster) / sizeof(taster[0]);
const int NO_PRESS_THRESHOLD = 200;  // darunter: kein Tastendruck (ggf. anpassen)

// Mittelwertbildung für stabilere ADC-Werte
int average() {
  long sum = 0;
  for (int i = 0; i < 10; i++) {
    sum += analogRead(pin);
    delay(2);
  }
  return sum / 10;
}

const char* lastButton = nullptr;

void setup() {
  Serial.begin(115200);

  // ESP32 ADC konfigurieren (12 Bit: 0–4095, 11dB: ca. 0–3.3V Bereich)
  analogReadResolution(12);
  analogSetAttenuation(ADC_11db);
}

void loop() {
  int analogValue = average();

  // Kein Tastendruck → Status zurücksetzen
  if (analogValue < NO_PRESS_THRESHOLD) {
    lastButton = nullptr;
    delay(50);
    return;
  }

  const char* currentButton = nullptr;
  int minV = 0, maxV = 0;

  // Passenden Taster anhand des Fensters finden
  for (int i = 0; i < ANZAHL_TASTER; i++) {
    if (analogValue >= taster[i].min_analog && analogValue <= taster[i].max_analog) {
      currentButton = taster[i].bezeichnung;
      minV = taster[i].min_analog;
      maxV = taster[i].max_analog;
      break;
    }
  }

  // Ausgabe nur bei Wechsel (verhindert Spam im seriellen Monitor)
  if (currentButton && currentButton != lastButton) {
    Serial.print(currentButton);
    Serial.print("\t|\t(");
    Serial.print(minV);
    Serial.print(" - ");
    Serial.print(maxV);
    Serial.print(")\t|\t");
    Serial.println(analogValue);

    lastButton = currentButton;
  }

  delay(50);
}
