#define pin 35  // ADC-Pin (GPIO35 ist input-only)

// --- RGB LED Pinbelegung (CYD Rückseite) ---
#define ledRedPin 4
#define ledGreenPin 16
#define ledBluePin 17

struct RGBColor {
  bool r;
  bool g;
  bool b;
};

struct Taster {
  int min_analog;
  int max_analog;
  const char* bezeichnung;
  RGBColor rgbColor;
};

// ADC-Fenster (aus realen Messwerten ermittelt)
Taster taster[] = {
  { 1233, 1306, "Taster 1", { false, true, false } },
  { 1865, 1880, "Taster 2", { true, true, false } },
  { 2555, 2630, "Taster 3", { false, false, true } },
  { 3213, 3235, "Taster 4", { true, true, true } },
  { 3690, 3720, "Taster 5", { true, false, false } }
};

const int ANZAHL_TASTER = sizeof(taster) / sizeof(taster[0]);
const int NO_PRESS_THRESHOLD = 200;  // darunter: kein Tastendruck (ggf. anpassen)

// Mittelwertbildung für stabilere ADC-Werte
int average() {
  long sum = 0;
  for (int i = 0; i < 10; i++) {
    sum += analogRead(pin);
    delay(2);
  }
  return sum / 10;
}

const char* lastButton = nullptr;

void setup() {
  Serial.begin(115200);

  // ESP32 ADC konfigurieren (12 Bit: 0–4095, 11dB: ca. 0–3.3V Bereich)
  analogReadResolution(12);
  analogSetAttenuation(ADC_11db);

  // RGB LED Pins als Ausgang definieren
  pinMode(ledRedPin, OUTPUT);
  pinMode(ledGreenPin, OUTPUT);
  pinMode(ledBluePin, OUTPUT);

  // Alle Farben ausschalten (Common Anode → HIGH = aus)
  digitalWrite(ledRedPin, HIGH);
  digitalWrite(ledGreenPin, HIGH);
  digitalWrite(ledBluePin, HIGH);
}

void loop() {
  int analogValue = average();

  // Kein Tastendruck → Status zurücksetzen
  if (analogValue < NO_PRESS_THRESHOLD) {
    lastButton = nullptr;
    delay(50);
    return;
  }

  const char* currentButton = nullptr;
  int minV = 0, maxV = 0;
  RGBColor color = { false, false, false };
  
  // Passenden Taster anhand des Fensters finden
  for (int i = 0; i < ANZAHL_TASTER; i++) {
    if (analogValue >= taster[i].min_analog && analogValue <= taster[i].max_analog) {
      currentButton = taster[i].bezeichnung;
      minV = taster[i].min_analog;
      maxV = taster[i].max_analog;
      color = taster[i].rgbColor;
      break;
    }
  }

  // Ausgabe nur bei Wechsel (verhindert Spam im seriellen Monitor)
  if (currentButton && currentButton != lastButton) {
    Serial.print(currentButton);
    Serial.print("\t|\t(");
    Serial.print(minV);
    Serial.print(" - ");
    Serial.print(maxV);
    Serial.print(")\t|\t");
    Serial.println(analogValue);

    lastButton = currentButton;
    setRgbLedValue(color.r, color.g, color.b);
  }

  delay(50);
}


// Funktion zum Setzen der RGB-Farbe
// true  = Farbe an
// false = Farbe aus
void setRgbLedValue(bool r, bool g, bool b) {

  // LOW schaltet LED EIN
  // HIGH schaltet LED AUS
  digitalWrite(ledRedPin, r ? LOW : HIGH);
  digitalWrite(ledGreenPin, g ? LOW : HIGH);
  digitalWrite(ledBluePin, b ? LOW : HIGH);
}