/*
  ================================================================
  Titel:      Cheap Yellow Display als Heizungs-Thermometer
  Projekt:    DS18B20 Tauchfühler mit LVGL Dashboard
  Autor:      Stefan Draeger
  Website:    https://draeger-it.blog
  Beitrag:    https://draeger-it.blog/cheap-yellow-display-als-heizungs-thermometer-ds18b20-tauchfuehler-anschliessen/
  Datum:      Februar 2026
  Plattform:  Cheap Yellow Display (ESP32-2432S028)
  ================================================================

  Beschreibung:
  ---------------------------------------------------------------
  Dieses Projekt zeigt, wie ein oder mehrere DS18B20 Temperatur-
  sensoren (wasserdichte Tauchfühler) am Cheap Yellow Display
  betrieben werden.

  Funktionen:
  - Auslesen von bis zu 5 DS18B20 Sensoren über OneWire (GPIO 27)
  - Eindeutige Adressierung über 64-Bit Sensor-ID
  - Temperaturmessung mit 11-Bit Auflösung
  - Anzeige der Messwerte auf dem Display via LVGL
  - Serielles Debugging über USB

  Ziel des Projekts ist es, ein kompaktes Heizungs-Thermometer
  mit Live-Anzeige zu realisieren, das sich flexibel erweitern lässt
  (z. B. Logging, Cloud-Anbindung, Grenzwertüberwachung).

  Hinweis:
  Für die Anzeige wird LVGL in Kombination mit TFT_eSPI verwendet.
  Die Display-Konfiguration muss zuvor korrekt eingerichtet sein.

  Weitere Details und Erweiterungen findest du im Blogbeitrag.
  ---------------------------------------------------------------
*/

#include <lvgl.h>
#include <TFT_eSPI.h>

#include <OneWire.h>
#include <DallasTemperature.h>

#define TFT_HOR_RES 240
#define TFT_VER_RES 320

#define TFT_ROTATION LV_DISPLAY_ROTATION_270

#define DRAW_BUF_SIZE (TFT_HOR_RES * TFT_VER_RES / 10 * (LV_COLOR_DEPTH / 8))
uint32_t draw_buf[DRAW_BUF_SIZE / 4];

// Farben wie in deinem Beispiel
#define COLOR_BG lv_color_hex(0x000000)
#define COLOR_WHITE lv_color_hex(0xFFFFFF)
#define COLOR_YELLOW lv_color_hex(0xFDD835)
#define COLOR_GRAY lv_color_hex(0x9E9E9E)

// ---------------- DS18B20 ----------------
#define sensorPin 27

OneWire oneWire(sensorPin);
DallasTemperature ds(&oneWire);

const int MAX_SENSORS = 5;
DeviceAddress sensorAddrs[MAX_SENSORS];

// ---------------- UI ----------------
lv_obj_t *titleLabel;
lv_obj_t *sensorLabels[MAX_SENSORS];

unsigned long lastRead = 0;
const int PAUSE = 1000;  // 1s Update ist praxisnah für Heizung
bool firstUpdateDone = false;

bool isValidAddress(const DeviceAddress addr) {
  for (uint8_t i = 0; i < 8; i++) {
    if (addr[i] != 0) return true;
  }
  return false;
}

void setupLVGL() {
  lv_init();

  lv_display_t *disp = lv_tft_espi_create(
    TFT_HOR_RES,
    TFT_VER_RES,
    draw_buf,
    sizeof(draw_buf));
  lv_display_set_rotation(disp, TFT_ROTATION);

  lv_obj_t *scr = lv_screen_active();
  lv_obj_set_style_bg_color(scr, COLOR_BG, 0);
  lv_obj_set_style_bg_opa(scr, LV_OPA_COVER, 0);
  lv_obj_set_style_pad_all(scr, 12, 0);
  lv_obj_set_scrollbar_mode(scr, LV_SCROLLBAR_MODE_OFF);

  // Titel
  titleLabel = lv_label_create(scr);
  lv_label_set_text(titleLabel, "DS18B20 Sensoren");
  lv_obj_set_style_text_color(titleLabel, COLOR_YELLOW, 0);
  lv_obj_set_style_text_font(titleLabel, &lv_font_montserrat_28, 0);
  lv_obj_align(titleLabel, LV_ALIGN_TOP_MID, 0, 0);

  // Sensorzeilen
  for (int i = 0; i < MAX_SENSORS; i++) {
    sensorLabels[i] = lv_label_create(scr);
    lv_obj_set_style_text_color(sensorLabels[i], COLOR_WHITE, 0);
    lv_obj_set_style_text_font(sensorLabels[i], &lv_font_montserrat_18, 0);
    lv_obj_align(sensorLabels[i], LV_ALIGN_TOP_LEFT, 0, 50 + i * 30);
    lv_label_set_text_fmt(sensorLabels[i], "Sensor %d: --.- \xC2\xB0C", i + 1);
  }

  // Kleiner Hinweis unten
  lv_obj_t *hint = lv_label_create(scr);
  lv_obj_set_style_text_color(hint, COLOR_GRAY, 0);
  lv_obj_set_style_text_font(hint, &lv_font_montserrat_12, 0);
  lv_label_set_text(hint, "Stefan Draeger - https://draeger-it.blog");
  lv_obj_align(hint, LV_ALIGN_BOTTOM_MID, 0, 0);
}

void setupSensors() {
  ds.begin();

  // Für schnellere Messungen: 11-bit (~375ms). Für Heizung völlig ausreichend.
  ds.setResolution(11);

  Serial.println("Suche nach DS18B20 Sensoren...");
  Serial.print("Anzahl gefundener Sensoren: ");
  Serial.println(ds.getDeviceCount());

  for (int i = 0; i < MAX_SENSORS; i++) {
    if (ds.getAddress(sensorAddrs[i], i)) {
      Serial.print("Sensor ");
      Serial.print(i + 1);
      Serial.print(" Adresse: ");
      for (uint8_t b = 0; b < 8; b++) {
        if (sensorAddrs[i][b] < 16) Serial.print("0");
        Serial.print(sensorAddrs[i][b], HEX);
      }
      Serial.println();
    } else {
      memset(sensorAddrs[i], 0, sizeof(DeviceAddress));
      Serial.print("Sensor ");
      Serial.print(i + 1);
      Serial.println(" wurde nicht gefunden!");
    }
  }
}

void updateDashboard() {
  ds.requestTemperatures();

  for (int i = 0; i < MAX_SENSORS; i++) {
    if (!isValidAddress(sensorAddrs[i])) {
      lv_label_set_text_fmt(sensorLabels[i], "Sensor %d: (nicht belegt)", i + 1);
      continue;
    }

    float tempC = ds.getTempC(sensorAddrs[i]);

    // DallasTemperature: -127 = disconnected / Fehler
    if (tempC < -100.0f || tempC > 150.0f) {
      lv_label_set_text_fmt(sensorLabels[i], "Sensor %d: Fehler", i + 1);
    } else {
      lv_label_set_text_fmt(sensorLabels[i], "Sensor %d: %.1f °C", i + 1, tempC);
      Serial.print("Sensor" + String(i + 1, DEC)+":");
      Serial.println(tempC);
    }
  }
}

void setup() {
  Serial.begin(9600);
  delay(50);

  setupLVGL();
  setupSensors();

  // Erste Anzeige sofort aktualisieren
  updateDashboard();
  firstUpdateDone = true;
}

void loop() {
  // exakt wie dein Beispiel
  lv_tick_inc(5);
  lv_timer_handler();

  unsigned long currentMillis = millis();
  if (!firstUpdateDone || currentMillis > (lastRead + PAUSE)) {
    lastRead = currentMillis;

    updateDashboard();  // UI + Serial optional
    // Optional: Debug-Ausgabe
    // Serial.println("Dashboard aktualisiert");
  }

  delay(5);
}