/*
Titel: ESP32 Edelmetall-Dashboard (LVGL + Touch + WiFi + JSON + NTP)

Beschreibung:
Dieses Projekt zeigt ein kompaktes Edelmetall-Dashboard auf einem ESP32 mit ILI9341-Display
(z. B. "Cheap Yellow Display" / ESP32-2432S028). Die Oberfläche wird mit LVGL gerendert,
Touch-Eingaben erfolgen über einen XPT2046-Controller. Die Edelmetallpreise werden zyklisch
per HTTP als JSON von [https://api.edelmetalle.de/public.json](https://api.edelmetalle.de/public.json) geladen und anschließend auf
dem Display angezeigt. Zusätzlich wird über NTP die aktuelle Uhrzeit bezogen und als
Zeitstempel im Footer ausgegeben. Per Touch-Buttons kann zwischen DE/US Anzeige sowie
einem manuellen Refresh gewechselt werden.

Autor: Stefan Draeger
Blogbeitrag: [https://draeger-it.blog/esp32-display-als-edelmetall-monitor-gold-und-silberpreis-in-echtzeit/](https://draeger-it.blog/esp32-display-als-edelmetall-monitor-gold-und-silberpreis-in-echtzeit/)
*/

#include <ArduinoJson.h>
#include <ArduinoJson.hpp>

// LVGL GUI Library
#include <lvgl.h>

// TFT Display Treiber (Bodmer TFT_eSPI)
#include <TFT_eSPI.h>

// Touchscreen Library (XPT2046)
#include <XPT2046_Touchscreen.h>

// WiFi + HTTP für API Abruf
#include <WiFi.h>
#include <HTTPClient.h>

// Zeit (NTP)
#include "time.h"

// WLAN Zugangsdaten (lokal ausgelagert)
#include "secrets.h"

/* ------------------------------------------------------------
Touch-Pins (XPT2046) – passend zu deinem Board-Setup
------------------------------------------------------------ */
#define XPT2046_IRQ 36
#define XPT2046_MOSI 32
#define XPT2046_MISO 39
#define XPT2046_CLK 25
#define XPT2046_CS 33

// Touch läuft über HSPI (separater SPI-Bus vom Display möglich)
SPIClass touchSPI(HSPI);
XPT2046_Touchscreen ts(XPT2046_CS, XPT2046_IRQ);

/* ------------------------------------------------------------
Display / LVGL Einstellungen
------------------------------------------------------------ */
#define TFT_HOR_RES 240
#define TFT_VER_RES 320
#define TFT_ROTATION LV_DISPLAY_ROTATION_270

// LVGL zeichnet in einen Puffer. 1/10 Bildschirm ist oft ein guter Kompromiss
#define DRAW_BUF_SIZE (TFT_HOR_RES * TFT_VER_RES / 10 * (LV_COLOR_DEPTH / 8))
uint32_t draw_buf[DRAW_BUF_SIZE / 4];

// Tick-Zeit (hier aktuell nicht genutzt, da du lv_tick_inc(5) nutzt)
static uint32_t last_tick = 0;

/* ------------------------------------------------------------
Farben (Dark Theme)
------------------------------------------------------------ */
#define COLOR_BG lv_color_hex(0x000000)
#define COLOR_TITLE lv_color_hex(0xFFFFFF)
#define COLOR_GOLD lv_color_hex(0xD4AF37)
#define COLOR_SILVER lv_color_hex(0xD9D9D9)
#define COLOR_PLATIN lv_color_hex(0xE5E4E2)     // platin: leicht warmes hellgrau
#define COLOR_PALLADIUM lv_color_hex(0xC0C0C0)  // palladium: etwas dunkler
#define COLOR_FOOTER lv_color_hex(0xAAAAAA)

/* ------------------------------------------------------------
Sprache (DE/US)
------------------------------------------------------------ */
enum Lang { LANG_DE,
            LANG_US };
static Lang g_lang = LANG_DE;

/* ------------------------------------------------------------
LVGL Objekt-Handles (werden später im Code gesetzt)
Damit apply_language() jederzeit Labels aktualisieren kann.
------------------------------------------------------------ */
static lv_obj_t *lbl_title;
static lv_obj_t *lbl_gold;
static lv_obj_t *lbl_silver;
static lv_obj_t *lbl_platinum;
static lv_obj_t *lbl_palladium;
static lv_obj_t *lbl_stamp;
static lv_obj_t *btn_lang;
static lv_obj_t *lbl_btn;
static lv_obj_t *btn_update;
static lv_obj_t *lbl_btn_update;

/* ------------------------------------------------------------
Forward Declarations (Prototypen)
------------------------------------------------------------ */
static void on_lang_btn(lv_event_t *e);
static void on_update_btn(lv_event_t *e);
static void apply_language();
static lv_obj_t *createLabel(lv_obj_t *parent, const char *text, lv_color_t color, int32_t posX, int32_t posY);

static lv_obj_t *create_gold_bar_icon(lv_obj_t *parent);
static lv_obj_t *create_silver_coin_icon(lv_obj_t *parent);
static lv_obj_t *create_platinum_bar_icon(lv_obj_t *parent);
static lv_obj_t *create_palladium_coin_icon(lv_obj_t *parent);

/* ------------------------------------------------------------
Touch-Kalibrierwerte
Diese Werte stammen aus einem Beispiel und können je nach Panel
leicht abweichen. Damit map() Rohwerte -> Pixelwerte umrechnet.
------------------------------------------------------------ */
#define TS_X_MIN 200
#define TS_X_MAX 3700
#define TS_Y_MIN 240
#define TS_Y_MAX 3800

/* ------------------------------------------------------------
API / Timing / NTP
------------------------------------------------------------ */
String metalApiUrl = "[https://api.edelmetalle.de/public.json](https://api.edelmetalle.de/public.json)";

// Timer für regelmäßige Updates (900000 ms = 15 Minuten)
unsigned long lastTime = 0;
unsigned long timerDelay = 900000;

// NTP
const char *ntpServer = "pool.ntp.org";
// Hinweis: gmtOffset_sec=60 wirkt ungewöhnlich (entspricht 1 Minute).
// Für Deutschland wäre im Winter 3600 (UTC+1) und im Sommer 7200 (UTC+2) üblich.
// Da du zusätzlich daylightOffset_sec=3600 setzt, funktioniert es evtl. dennoch.
// Wenn du absolute Korrektheit willst, stelle gmtOffset_sec=3600 und daylightOffset_sec=3600.
const long gmtOffset_sec = 60;
const int daylightOffset_sec = 3600;

// Zeitstrings für Footer (DE/US Format)
char deTimeStr[25];
char usTimeStr[25];

/* ------------------------------------------------------------
Daten-Container (werden aus JSON befüllt)
------------------------------------------------------------ */
double gold_usd = 0.0d;
double silber_usd = 0.0d;
double platin_usd = 0.0d;
double palladium_usd = 0.0d;
double gold_eur = 0.0d;
double silber_eur = 0.0d;
double platin_eur = 0.0d;
double palladium_eur = 0.0d;

/* ------------------------------------------------------------
Touch-Callback für LVGL
Wird bei jedem LVGL Input Poll aufgerufen.
------------------------------------------------------------ */
static void my_touch_read(lv_indev_t *indev, lv_indev_data_t *data) {
  // Prüfen, ob Touch-IRQ aktiv und Touch gültig
  if (ts.tirqTouched() && ts.touched()) {
    TS_Point p = ts.getPoint();  // Rohwerte vom Touchcontroller


    // Rohwerte -> Pixel (Mapping anhand Kalibrierwerte)
    int px = map(p.x, TS_X_MIN, TS_X_MAX, 1, TFT_HOR_RES);
    int py = map(p.y, TS_Y_MIN, TS_Y_MAX, 1, TFT_VER_RES);

    // Sicherheit: innerhalb Bildschirm bleiben
    if (px < 0) px = 0;
    if (px > TFT_HOR_RES - 1) px = TFT_HOR_RES - 1;
    if (py < 0) py = 0;
    if (py > TFT_VER_RES - 1) py = TFT_VER_RES - 1;

    // LVGL mitteilen: Touch pressed + Koordinate
    data->state = LV_INDEV_STATE_PRESSED;
    data->point.x = px;
    data->point.y = py;


  } else {
    // Nicht berührt
    data->state = LV_INDEV_STATE_RELEASED;
  }
}

/* ------------------------------------------------------------
Forward: Daten-Funktionen
------------------------------------------------------------ */
void refreshData();
void handleJsonPayload(String json);
void setTimestamps();

/* ------------------------------------------------------------
Setup: Hardware + GUI initialisieren
------------------------------------------------------------ */
void setup() {
  Serial.begin(9600);

  // ---------- WiFi verbinden ----------
  Serial.println("\nVerbinde mit WiFi...");
  WiFi.begin(ssid, password);

  // Blockierend warten, bis verbunden
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println("\nVerbunden!");
  Serial.print("IP Adresse: ");
  Serial.println(WiFi.localIP());

  // ---------- NTP Zeit initialisieren ----------
  configTime(gmtOffset_sec, daylightOffset_sec, ntpServer);

  // ---------- LVGL initialisieren ----------
  lv_init();

  // ---------- Touch SPI + Touch init ----------
  touchSPI.begin(XPT2046_CLK, XPT2046_MISO, XPT2046_MOSI, XPT2046_CS);
  ts.begin(touchSPI);
  ts.setRotation(2);

  // ---------- Display init (TFT_eSPI über LVGL Helper) ----------
  lv_display_t *disp = lv_tft_espi_create(TFT_HOR_RES, TFT_VER_RES, draw_buf, sizeof(draw_buf));
  lv_display_set_rotation(disp, TFT_ROTATION);

  // ---------- Input Device (Touch) an LVGL binden ----------
  lv_indev_t *indev = lv_indev_create();
  lv_indev_set_type(indev, LV_INDEV_TYPE_POINTER);
  lv_indev_set_read_cb(indev, my_touch_read);

  // ---------- Screen Style ----------
  lv_obj_t *scr = lv_screen_active();
  lv_obj_set_style_bg_color(scr, COLOR_BG, 0);
  lv_obj_set_style_bg_opa(scr, LV_OPA_COVER, 0);
  lv_obj_set_style_pad_all(scr, 12, 0);
  lv_obj_set_scrollbar_mode(scr, LV_SCROLLBAR_MODE_OFF);

  // ---------- Titel ----------
  lbl_title = lv_label_create(scr);
  lv_label_set_text(lbl_title, "Edelmetall Dashboard");
  lv_obj_set_style_text_color(lbl_title, COLOR_TITLE, 0);
  lv_obj_set_style_text_font(lbl_title, &lv_font_montserrat_20, 0);
  lv_obj_align(lbl_title, LV_ALIGN_TOP_MID, 0, 0);

  // ---------- Footer Container (unten) ----------
  lv_obj_t *footer = lv_obj_create(scr);
  lv_obj_remove_style_all(footer);
  lv_obj_set_height(footer, LV_SIZE_CONTENT);
  lv_obj_set_style_pad_all(footer, 0, 0);
  lv_obj_set_width(footer, LV_PCT(100));
  lv_obj_set_style_pad_row(footer, 4, 0);
  lv_obj_set_flex_flow(footer, LV_FLEX_FLOW_COLUMN);
  lv_obj_set_flex_align(footer, LV_FLEX_ALIGN_CENTER, LV_FLEX_ALIGN_CENTER, LV_FLEX_ALIGN_CENTER);
  lv_obj_align(footer, LV_ALIGN_BOTTOM_MID, 0, 0);

  // Zeitstempel-Label (wird später dynamisch gesetzt)
  lbl_stamp = lv_label_create(footer);
  lv_label_set_text(lbl_stamp, "Stand: 16.01.2026 - 16:46 Uhr");
  lv_obj_set_style_text_color(lbl_stamp, COLOR_FOOTER, 0);
  lv_obj_set_style_text_font(lbl_stamp, &lv_font_montserrat_12, 0);

  // Credit-Label
  lv_obj_t *credit = lv_label_create(footer);
  lv_label_set_text(credit, "Stefan Draeger - [https://draeger-it.blog](https://draeger-it.blog)");
  lv_obj_set_style_text_color(credit, COLOR_FOOTER, 0);
  lv_obj_set_style_text_font(credit, &lv_font_montserrat_12, 0);

  // ---------- Toolbar im Footer (Buttons) ----------
  lv_obj_t *toolbar = lv_obj_create(footer);
  lv_obj_remove_style_all(toolbar);
  lv_obj_set_style_pad_column(toolbar, 10, 0);  // Abstand zwischen den Buttons (Flex Row)
  lv_obj_set_height(toolbar, LV_SIZE_CONTENT);
  lv_obj_set_style_pad_all(toolbar, 0, 0);
  lv_obj_set_width(toolbar, LV_PCT(100));
  lv_obj_set_style_pad_row(toolbar, 4, 0);
  lv_obj_set_flex_flow(toolbar, LV_FLEX_FLOW_ROW);
  lv_obj_set_flex_align(toolbar, LV_FLEX_ALIGN_CENTER, LV_FLEX_ALIGN_CENTER, LV_FLEX_ALIGN_CENTER);
  lv_obj_align(toolbar, LV_ALIGN_BOTTOM_MID, 0, 0);

  // ---------- Language Button ----------
  btn_lang = lv_btn_create(toolbar);
  lv_obj_set_size(btn_lang, 52, 18);
  lv_obj_set_pos(btn_lang, 0, 0);
  lv_obj_add_event_cb(btn_lang, on_lang_btn, LV_EVENT_PRESSED, NULL);
  lv_obj_set_style_bg_color(btn_lang, lv_color_hex(0x202020), 0);
  lv_obj_set_style_border_width(btn_lang, 1, 0);
  lv_obj_set_style_border_color(btn_lang, lv_color_hex(0x444444), 0);

  lbl_btn = lv_label_create(btn_lang);
  lv_obj_center(lbl_btn);  // Label zentrieren
  lv_obj_set_style_text_font(lbl_btn, &lv_font_montserrat_10, 0);
  lv_obj_set_style_text_color(lbl_btn, lv_color_hex(0xFFFFFF), 0);

  // ---------- Update Button ----------
  btn_update = lv_btn_create(toolbar);
  lv_obj_set_size(btn_update, 105, 18);
  lv_obj_set_pos(btn_update, 0, 0);
  lv_obj_add_event_cb(btn_update, on_update_btn, LV_EVENT_PRESSED, NULL);
  lv_obj_set_style_bg_color(btn_update, lv_color_hex(0x202020), 0);
  lv_obj_set_style_border_width(btn_update, 1, 0);
  lv_obj_set_style_border_color(btn_update, lv_color_hex(0x444444), 0);

  lbl_btn_update = lv_label_create(btn_update);
  lv_obj_center(lbl_btn_update);  // Label zentrieren
  lv_obj_set_style_text_font(lbl_btn_update, &lv_font_montserrat_10, 0);
  lv_obj_set_style_text_color(lbl_btn_update, lv_color_hex(0xFFFFFF), 0);

  // Layout updaten, damit Größen/Positionen sauber berechnet sind
  lv_obj_update_layout(btn_lang);
  lv_obj_update_layout(btn_update);
  lv_obj_update_layout(lbl_title);
  lv_obj_update_layout(footer);

  // ---------- Content Bereich (zwischen Title und Footer) ----------
  int y_content = lv_obj_get_y(lbl_title) + lv_obj_get_height(lbl_title) + 12;
  int h_content = lv_obj_get_y(footer) - y_content - 2;

  lv_obj_t *content = lv_obj_create(scr);
  lv_obj_remove_style_all(content);
  lv_obj_set_pos(content, 15, y_content + 8);
  lv_obj_set_size(content, lv_obj_get_width(scr), h_content);
  lv_obj_set_scroll_dir(content, LV_DIR_NONE);
  lv_obj_set_scrollbar_mode(content, LV_SCROLLBAR_MODE_OFF);

  // ---------- Zeilen: Icons + Labels ----------
  lv_obj_t *gold_icon = create_gold_bar_icon(content);
  lv_obj_set_pos(gold_icon, 0, 2);
  lbl_gold = createLabel(content, "Goldpreis: 3976,16 EUR", COLOR_GOLD, 24, 0);

  lv_obj_t *silver_icon = create_silver_coin_icon(content);
  lv_obj_set_pos(silver_icon, 2, 30);
  lbl_silver = createLabel(content, "Silberpreis: 79,5325 EUR", COLOR_SILVER, 24, 28);

  lv_obj_t *platinum_icon = create_platinum_bar_icon(content);
  lv_obj_set_pos(platinum_icon, 0, 58);
  lbl_platinum = createLabel(content, "Platinpreis: 0,00 EUR", COLOR_PLATIN, 24, 56);

  lv_obj_t *palladium_icon = create_palladium_coin_icon(content);
  lv_obj_set_pos(palladium_icon, 2, 86);
  lbl_palladium = createLabel(content, "Palladiumpreis: 0,00 EUR", COLOR_PALLADIUM, 24, 84);

  // ---------- Initialer Datenabruf + UI refresh ----------
  refreshData();
  apply_language();
}

/* ------------------------------------------------------------
Helper: Label erstellen (Text, Farbe, Font, Position)
------------------------------------------------------------ */
static lv_obj_t *createLabel(lv_obj_t *parent, const char *text, lv_color_t color, int32_t posX, int32_t posY) {
  lv_obj_t *label = lv_label_create(parent);
  lv_label_set_text(label, text);
  lv_obj_set_style_text_color(label, color, 0);
  lv_obj_set_style_text_font(label, &lv_font_montserrat_18, 0);
  lv_obj_set_pos(label, posX, posY);
  return label;
}

/* ------------------------------------------------------------
Loop: LVGL + regelmäßige Updates
------------------------------------------------------------ */
void loop() {
  // LVGL "arbeiten lassen"
  lv_timer_handler();

  // Wichtig: LVGL Zeit fortschreiben (Tick)
  // Du verwendest hier ein fixes 5ms Raster.
  lv_tick_inc(5);
  delay(5);

  // Periodischer API Refresh
  if ((millis() - lastTime) > timerDelay) {
    refreshData();
  }
}

/* ------------------------------------------------------------
refreshData(): Zeitstempel setzen + HTTP Request + JSON parsen
Danach UI aktualisieren (apply_language)
------------------------------------------------------------ */
void refreshData() {
  if (WiFi.status() == WL_CONNECTED) {
    setTimestamps();


    HTTPClient http;
    http.begin(metalApiUrl.c_str());
    int httpResponseCode = http.GET();

    if (httpResponseCode == 200) {
      String payload = http.getString();
      handleJsonPayload(payload);
    } else if (httpResponseCode == 304) {
      Serial.println("Keine neuen Daten!");
    } else {
      Serial.print("HTTP Error: ");
      Serial.println(httpResponseCode);
    }
    http.end();

  } else {
    Serial.println("WiFi nicht verbunden – kein Refresh möglich.");
  }

  lastTime = millis();
  apply_language();  // UI immer neu setzen (DE/US)
}

/* ------------------------------------------------------------
Zeitstempel (NTP) in DE/US Format erzeugen
------------------------------------------------------------ */
void setTimestamps() {
  struct tm timeinfo;
  if (!getLocalTime(&timeinfo)) {
    Serial.println("Failed to obtain time");
    return;
  }

  // DE: 16.01.2026 - 16:46
  strftime(deTimeStr, sizeof(deTimeStr), "%d.%m.%Y - %H:%M", &timeinfo);

  // US: 2026-01-16 - 16:46
  strftime(usTimeStr, sizeof(usTimeStr), "%Y-%m-%d - %H:%M", &timeinfo);
}

/* ------------------------------------------------------------
JSON Payload verarbeiten
Achtung: StaticJsonDocument Größe ggf. erhöhen, falls API wächst.
------------------------------------------------------------ */
void handleJsonPayload(String json) {
  StaticJsonDocument<400> doc;
  DeserializationError error = deserializeJson(doc, json);

  if (error) {
    Serial.print(F("deserializeJson() failed: "));
    Serial.println(error.f_str());
    return;
  }

  // USD Werte
  gold_usd = doc["gold_usd"];
  silber_usd = doc["silber_usd"];
  platin_usd = doc["platin_usd"];
  palladium_usd = doc["palladium_usd"];

  // EUR Werte
  gold_eur = doc["gold_eur"];
  silber_eur = doc["silber_eur"];
  platin_eur = doc["platin_eur"];
  palladium_eur = doc["palladium_eur"];
}

/* ------------------------------------------------------------
Sprache anwenden: Labels dynamisch setzen (DE/US)
Wichtig: nutzt die aktuell gespeicherten Variablen (EUR/USD)
------------------------------------------------------------ */
static void apply_language() {
  if (g_lang == LANG_DE) {
    lv_label_set_text(lbl_title, "Edelmetall Dashboard");

    char goldEurlabelText[64];
    snprintf(goldEurlabelText, sizeof(goldEurlabelText), "Goldpreis: %.2f EUR", gold_eur);
    lv_label_set_text(lbl_gold, goldEurlabelText);

    char silberEurlabelText[64];
    snprintf(silberEurlabelText, sizeof(silberEurlabelText), "Silberpreis: %.2f EUR", silber_eur);
    lv_label_set_text(lbl_silver, silberEurlabelText);

    char platinumEurlabelText[64];
    snprintf(platinumEurlabelText, sizeof(platinumEurlabelText), "Platinpreis: %.2f EUR", platin_eur);
    lv_label_set_text(lbl_platinum, platinumEurlabelText);

    char palladiumEurlabelText[64];
    snprintf(palladiumEurlabelText, sizeof(palladiumEurlabelText), "Palladiumpreis: %.2f EUR", palladium_eur);
    lv_label_set_text(lbl_palladium, palladiumEurlabelText);

    char labelText[40];
    snprintf(labelText, sizeof(labelText), "Stand: %s", deTimeStr);
    lv_label_set_text(lbl_stamp, labelText);

    lv_label_set_text(lbl_btn, "DE");
    lv_label_set_text(lbl_btn_update, "Aktualisieren");

  } else {
    lv_label_set_text(lbl_title, "Precious Metals Dashboard");

    char goldUsdlabelText[64];
    snprintf(goldUsdlabelText, sizeof(goldUsdlabelText), "Gold price: %.2f USD", gold_usd);
    lv_label_set_text(lbl_gold, goldUsdlabelText);

    char silberUsdlabelText[64];
    snprintf(silberUsdlabelText, sizeof(silberUsdlabelText), "Silver price: %.2f USD", silber_usd);
    lv_label_set_text(lbl_silver, silberUsdlabelText);

    char platinumUsdlabelText[64];
    snprintf(platinumUsdlabelText, sizeof(platinumUsdlabelText), "Platinum price: %.2f USD", platin_usd);
    lv_label_set_text(lbl_platinum, platinumUsdlabelText);

    char palladiumUsdlabelText[64];
    snprintf(palladiumUsdlabelText, sizeof(palladiumUsdlabelText), "Palladium price: %.2f USD", palladium_usd);
    lv_label_set_text(lbl_palladium, palladiumUsdlabelText);

    char labelText[40];
    snprintf(labelText, sizeof(labelText), "Updated: %s", usTimeStr);
    lv_label_set_text(lbl_stamp, labelText);

    lv_label_set_text(lbl_btn, "US");
    lv_label_set_text(lbl_btn_update, "Update");
  }
}

/* ------------------------------------------------------------
Event: Sprachbutton
------------------------------------------------------------ */
static void on_lang_btn(lv_event_t *e) {
  // Sprache toggeln
  if (g_lang == LANG_DE) g_lang = LANG_US;
  else g_lang = LANG_DE;

  apply_language();
}

/* ------------------------------------------------------------
Event: Update Button (manueller Refresh)
------------------------------------------------------------ */
static void on_update_btn(lv_event_t *e) {
  refreshData();
}

/* ------------------------------------------------------------
Icons: Gold / Silber / Platin / Palladium (simple shapes)
------------------------------------------------------------ */
static lv_obj_t *create_gold_bar_icon(lv_obj_t *parent) {
  lv_obj_t *bar = lv_obj_create(parent);
  lv_obj_remove_style_all(bar);
  lv_obj_set_size(bar, 16, 12);

  lv_obj_set_style_bg_color(bar, lv_color_hex(0xD4AF37), 0);
  lv_obj_set_style_bg_opa(bar, LV_OPA_COVER, 0);
  lv_obj_set_style_radius(bar, 3, 0);

  lv_obj_set_style_border_width(bar, 1, 0);
  lv_obj_set_style_border_color(bar, lv_color_hex(0xB38B2E), 0);

  lv_obj_t *shine = lv_obj_create(bar);
  lv_obj_remove_style_all(shine);
  lv_obj_set_size(shine, 10, 3);
  lv_obj_align(shine, LV_ALIGN_TOP_LEFT, 2, 2);
  lv_obj_set_style_bg_color(shine, lv_color_hex(0xFFE08A), 0);
  lv_obj_set_style_bg_opa(shine, LV_OPA_COVER, 0);
  lv_obj_set_style_radius(shine, 2, 0);

  lv_obj_t *shadow = lv_obj_create(bar);
  lv_obj_remove_style_all(shadow);
  lv_obj_set_size(shadow, 12, 3);
  lv_obj_align(shadow, LV_ALIGN_BOTTOM_RIGHT, -2, -2);
  lv_obj_set_style_bg_color(shadow, lv_color_hex(0xA77A1E), 0);
  lv_obj_set_style_bg_opa(shadow, LV_OPA_COVER, 0);
  lv_obj_set_style_radius(shadow, 2, 0);

  return bar;
}

static lv_obj_t *create_silver_coin_icon(lv_obj_t *parent) {
  lv_obj_t *coin = lv_obj_create(parent);
  lv_obj_remove_style_all(coin);
  lv_obj_set_size(coin, 12, 12);

  lv_obj_set_style_bg_color(coin, lv_color_hex(0xD9D9D9), 0);
  lv_obj_set_style_bg_opa(coin, LV_OPA_COVER, 0);
  lv_obj_set_style_radius(coin, LV_RADIUS_CIRCLE, 0);

  lv_obj_set_style_border_width(coin, 1, 0);
  lv_obj_set_style_border_color(coin, lv_color_hex(0xA9A9A9), 0);

  lv_obj_t *shine = lv_obj_create(coin);
  lv_obj_remove_style_all(shine);
  lv_obj_set_size(shine, 6, 3);
  lv_obj_align(shine, LV_ALIGN_TOP_LEFT, 2, 2);
  lv_obj_set_style_bg_color(shine, lv_color_hex(0xFFFFFF), 0);
  lv_obj_set_style_bg_opa(shine, LV_OPA_60, 0);
  lv_obj_set_style_radius(shine, LV_RADIUS_CIRCLE, 0);

  lv_obj_t *shadow = lv_obj_create(coin);
  lv_obj_remove_style_all(shadow);
  lv_obj_set_size(shadow, 7, 3);
  lv_obj_align(shadow, LV_ALIGN_BOTTOM_RIGHT, -2, -2);
  lv_obj_set_style_bg_color(shadow, lv_color_hex(0x8C8C8C), 0);
  lv_obj_set_style_bg_opa(shadow, LV_OPA_50, 0);
  lv_obj_set_style_radius(shadow, LV_RADIUS_CIRCLE, 0);

  lv_obj_t *ring = lv_obj_create(coin);
  lv_obj_remove_style_all(ring);
  lv_obj_set_size(ring, 8, 8);
  lv_obj_center(ring);
  lv_obj_set_style_radius(ring, LV_RADIUS_CIRCLE, 0);
  lv_obj_set_style_border_width(ring, 1, 0);
  lv_obj_set_style_border_color(ring, lv_color_hex(0xBEBEBE), 0);
  lv_obj_set_style_bg_opa(ring, LV_OPA_TRANSP, 0);

  return coin;
}

static lv_obj_t *create_platinum_bar_icon(lv_obj_t *parent) {
  lv_obj_t *bar = lv_obj_create(parent);
  lv_obj_remove_style_all(bar);
  lv_obj_set_size(bar, 16, 12);

  lv_obj_set_style_bg_color(bar, lv_color_hex(0xE5E4E2), 0);
  lv_obj_set_style_bg_opa(bar, LV_OPA_COVER, 0);
  lv_obj_set_style_radius(bar, 3, 0);

  lv_obj_set_style_border_width(bar, 1, 0);
  lv_obj_set_style_border_color(bar, lv_color_hex(0xAFAEAD), 0);

  lv_obj_t *shine = lv_obj_create(bar);
  lv_obj_remove_style_all(shine);
  lv_obj_set_size(shine, 10, 3);
  lv_obj_align(shine, LV_ALIGN_TOP_LEFT, 2, 2);
  lv_obj_set_style_bg_color(shine, lv_color_hex(0xFFFFFF), 0);
  lv_obj_set_style_bg_opa(shine, LV_OPA_70, 0);
  lv_obj_set_style_radius(shine, 2, 0);

  lv_obj_t *shadow = lv_obj_create(bar);
  lv_obj_remove_style_all(shadow);
  lv_obj_set_size(shadow, 12, 3);
  lv_obj_align(shadow, LV_ALIGN_BOTTOM_RIGHT, -2, -2);
  lv_obj_set_style_bg_color(shadow, lv_color_hex(0x8F8E8D), 0);
  lv_obj_set_style_bg_opa(shadow, LV_OPA_60, 0);
  lv_obj_set_style_radius(shadow, 2, 0);

  return bar;
}

static lv_obj_t *create_palladium_coin_icon(lv_obj_t *parent) {
  lv_obj_t *coin = lv_obj_create(parent);
  lv_obj_remove_style_all(coin);
  lv_obj_set_size(coin, 12, 12);

  lv_obj_set_style_bg_color(coin, lv_color_hex(0xC0C0C0), 0);
  lv_obj_set_style_bg_opa(coin, LV_OPA_COVER, 0);
  lv_obj_set_style_radius(coin, LV_RADIUS_CIRCLE, 0);

  lv_obj_set_style_border_width(coin, 1, 0);
  lv_obj_set_style_border_color(coin, lv_color_hex(0x8A8A8A), 0);

  lv_obj_t *shine = lv_obj_create(coin);
  lv_obj_remove_style_all(shine);
  lv_obj_set_size(shine, 6, 3);
  lv_obj_align(shine, LV_ALIGN_TOP_LEFT, 2, 2);
  lv_obj_set_style_bg_color(shine, lv_color_hex(0xFFFFFF), 0);
  lv_obj_set_style_bg_opa(shine, LV_OPA_50, 0);
  lv_obj_set_style_radius(shine, LV_RADIUS_CIRCLE, 0);

  lv_obj_t *ring = lv_obj_create(coin);
  lv_obj_remove_style_all(ring);
  lv_obj_set_size(ring, 8, 8);
  lv_obj_center(ring);
  lv_obj_set_style_radius(ring, LV_RADIUS_CIRCLE, 0);
  lv_obj_set_style_border_width(ring, 1, 0);
  lv_obj_set_style_border_color(ring, lv_color_hex(0xB0B0B0), 0);
  lv_obj_set_style_bg_opa(ring, LV_OPA_TRANSP, 0);

  return coin;
}
