#include <lvgl.h>
#include <TFT_eSPI.h>
#include <XPT2046_Touchscreen.h>

#define XPT2046_IRQ 36
#define XPT2046_MOSI 32
#define XPT2046_MISO 39
#define XPT2046_CLK 25
#define XPT2046_CS 33

SPIClass touchSPI(HSPI);
XPT2046_Touchscreen ts(XPT2046_CS, XPT2046_IRQ);

#define TFT_HOR_RES 240
#define TFT_VER_RES 320
#define TFT_ROTATION LV_DISPLAY_ROTATION_270

static void on_btn_press(lv_event_t *e);

#define TS_X_MIN 200
#define TS_X_MAX 3700
#define TS_Y_MIN 240
#define TS_Y_MAX 3800

#define DRAW_BUF_SIZE (TFT_HOR_RES * TFT_VER_RES / 10 * (LV_COLOR_DEPTH / 8))
uint32_t draw_buf[DRAW_BUF_SIZE / 4];

#define COLOR_BLACK lv_color_hex(0x000000)
#define COLOR_GOLD lv_color_hex(0xD4AF37)
#define COLOR_SILVER lv_color_hex(0xD9D9D9)
#define COLOR_WHITE lv_color_hex(0xFFFFFF)
#define COLOR_BLUE lv_color_hex(0x1E88E5)

volatile int tapCount = 0;

lv_obj_t *lbl_counter;

static void my_touch_read(lv_indev_t *indev, lv_indev_data_t *data) {
  // Prüfen, ob Touch-IRQ aktiv und Touch gültig
  if (ts.tirqTouched() && ts.touched()) {
    TS_Point p = ts.getPoint();  // Rohwerte vom Touchcontroller


    // Rohwerte -> Pixel (Mapping anhand Kalibrierwerte)
    int px = map(p.x, TS_X_MIN, TS_X_MAX, 1, TFT_HOR_RES);
    int py = map(p.y, TS_Y_MIN, TS_Y_MAX, 1, TFT_VER_RES);

    // Sicherheit: innerhalb Bildschirm bleiben
    if (px < 0) px = 0;
    if (px > TFT_HOR_RES - 1) px = TFT_HOR_RES - 1;
    if (py < 0) py = 0;
    if (py > TFT_VER_RES - 1) py = TFT_VER_RES - 1;

    // LVGL mitteilen: Touch pressed + Koordinate
    data->state = LV_INDEV_STATE_PRESSED;
    data->point.x = px;
    data->point.y = py;


  } else {
    // Nicht berührt
    data->state = LV_INDEV_STATE_RELEASED;
  }
}

void setup() {
  Serial.begin(9600);
  lv_init();

  touchSPI.begin(XPT2046_CLK, XPT2046_MISO, XPT2046_MOSI, XPT2046_CS);
  ts.begin(touchSPI);
  ts.setRotation(2);

  lv_display_t *disp = lv_tft_espi_create(TFT_HOR_RES, TFT_VER_RES, draw_buf, sizeof(draw_buf));
  lv_display_set_rotation(disp, TFT_ROTATION);

  lv_indev_t *indev = lv_indev_create();
  lv_indev_set_type(indev, LV_INDEV_TYPE_POINTER);
  lv_indev_set_read_cb(indev, my_touch_read);

  lv_obj_t *scr = lv_screen_active();
  lv_obj_set_style_bg_color(scr, COLOR_BLACK, 0);
  lv_obj_set_style_bg_opa(scr, LV_OPA_COVER, 0);
  lv_obj_set_style_pad_all(scr, 12, 0);
  lv_obj_set_scrollbar_mode(scr, LV_SCROLLBAR_MODE_OFF);

  lv_obj_t *btn = lv_btn_create(scr);
  lv_obj_set_size(btn, 80, 25);
  lv_obj_set_pos(btn, 0, 0);
  lv_obj_add_event_cb(btn, on_btn_press, LV_EVENT_PRESSED, NULL);
  lv_obj_set_style_bg_color(btn, COLOR_GOLD, 0);
  lv_obj_set_style_border_width(btn, 1, 0);
  lv_obj_set_style_border_color(btn, COLOR_SILVER, 0);

  lv_obj_t *lbl_btn = lv_label_create(btn);
  lv_obj_center(lbl_btn);
  lv_label_set_text(lbl_btn, "Klick mich!");
  lv_obj_set_style_text_font(lbl_btn, &lv_font_montserrat_10, 0);
  lv_obj_set_style_text_color(lbl_btn, COLOR_BLACK, 0);

  lbl_counter = lv_label_create(scr);
  lv_obj_set_style_text_font(lbl_counter, &lv_font_montserrat_28, 0);
  lv_obj_set_style_text_color(lbl_counter, COLOR_BLUE, 0);
  lv_obj_set_pos(lbl_counter, 120, 100);

  handlePressBtn();
}

void loop() {
  lv_timer_handler();
  lv_tick_inc(5);
  delay(5);
}

void handlePressBtn() {
  tapCount++;
  String s = String(tapCount, DEC);
  lv_label_set_text(lbl_counter, s.c_str());
}

static void on_btn_press(lv_event_t *e) {
  handlePressBtn();
}
