# ============================================================
# ESP32 DC-Motor steuern mit MicroPython
# PWM-Drehzahlregelung über Drehpotentiometer
# Geschwindigkeit wird zusätzlich über 5 LEDs visualisiert
#
# Autor: Stefan Draeger
# ============================================================

from machine import Pin, ADC, PWM
import time

# ==============================
# Pinbelegung
# ==============================
led1Pin1 = 18
led1Pin2 = 23
led1Pin3 = 12
led1Pin4 = 17
led1Pin5 = 26

drehPotiPin = 2
motorPin = 4

# ==============================
# LEDs initialisieren
# ==============================
led_pins = [
    Pin(led1Pin1, Pin.OUT),
    Pin(led1Pin2, Pin.OUT),
    Pin(led1Pin3, Pin.OUT),
    Pin(led1Pin4, Pin.OUT),
    Pin(led1Pin5, Pin.OUT),
]

MAX_LEDS = len(led_pins)

# ==============================
# Drehpotentiometer (ADC)
# ==============================
poti = ADC(Pin(drehPotiPin))
poti.atten(ADC.ATTN_11DB)      # Messbereich bis ca. 3,3 V
poti.width(ADC.WIDTH_12BIT)    # Wertebereich 0–4095

# ==============================
# PWM für den Motor
# ==============================
motor_pwm = PWM(Pin(motorPin))
motor_pwm.freq(1000)           # 1 kHz PWM-Frequenz

# ==============================
# Hilfsfunktionen
# ==============================
def reset_leds():
    """Schaltet alle LEDs aus"""
    for led in led_pins:
        led.off()

def display_speed(value):
    """
    Zeigt die Geschwindigkeit über LEDs an
    ADC-Wert (0–4095) → LED-Stufe (0–4)
    """
    led_index = int(value / 4095 * (MAX_LEDS - 1))
    reset_leds()
    led_pins[led_index].on()

def set_motor_speed(value):
    """
    Setzt die Motorgeschwindigkeit per PWM
    ADC-Wert (0–4095) → PWM-Duty (0–1023)
    """
    duty = int(value / 4095 * 1023)
    motor_pwm.duty(duty)

# ==============================
# Start-LED-Test
# ==============================
for led in led_pins:
    led.on()
    time.sleep(0.2)
reset_leds()

# ==============================
# Hauptschleife
# ==============================
last_value = -1

while True:
    poti_value = poti.read()

    # Nur reagieren, wenn sich der Wert ändert
    if abs(poti_value - last_value) > 10:
        display_speed(poti_value)
        set_motor_speed(poti_value)
        last_value = poti_value

    time.sleep(0.05)
 
